"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
var _path = _interopRequireDefault(require("path"));
var _webpack = require("next/dist/compiled/webpack/webpack");
var _constants = require("../../../lib/constants");
function _interopRequireDefault(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const PLUGIN_NAME = "FlightTypesPlugin";
function createTypeGuardFile(fullPath, relativePath, options) {
    return `// File: ${fullPath}
import * as entry from '${relativePath}'
type TEntry = typeof entry

check<IEntry, TEntry>(entry)

type PageParams = Record<string, string>
interface PageProps {
  params?: PageParams
  searchParams?: Record<string, string | string[]>
}
interface LayoutProps {
  children: React.ReactNode
  params?: PageParams
}

type PageComponent = (props: PageProps) => React.ReactNode | Promise<React.ReactNode>
type LayoutComponent = (props: LayoutProps) => React.ReactNode | Promise<React.ReactNode>

interface IEntry {
  ${options.type === "layout" ? `default: LayoutComponent` : `default: PageComponent`}
  config?: {}
  generateStaticParams?: (params?: PageParams) => Promise<any[]>
  revalidate?: RevalidateRange<TEntry> | false
  dynamic?: 'auto' | 'force-dynamic' | 'error' | 'force-static'
  dynamicParams?: boolean
  fetchCache?: 'auto' | 'force-no-store' | 'only-no-store' | 'default-no-store' | 'default-cache' | 'only-cache' | 'force-cache'
  preferredRegion?: 'auto' | 'home' | 'edge'
  ${options.type === "page" ? "runtime?: 'nodejs' | 'experimental-edge'" : ""}
}

// =============
// Utility types
type RevalidateRange<T> = T extends { revalidate: any } ? NonNegative<T['revalidate']> : never
type Impossible<K extends keyof any> = { [P in K]: never }
function check<Base, T extends Base>(_mod: T & Impossible<Exclude<keyof T, keyof Base>>): void {}

// https://github.com/sindresorhus/type-fest
type Numeric = number | bigint
type Zero = 0 | 0n
type Negative<T extends Numeric> = T extends Zero ? never : \`\${T}\` extends \`-\${string}\` ? T : never
type NonNegative<T extends Numeric> = T extends Zero ? T : Negative<T> extends never ? T : '__invalid_negative_number__'
`;
}
class FlightTypesPlugin {
    constructor(options){
        this.dir = options.dir;
        this.appDir = options.appDir;
        this.dev = options.dev;
        this.isEdgeServer = options.isEdgeServer;
    }
    apply(compiler) {
        const assetPrefix = this.dev ? ".." : this.isEdgeServer ? ".." : "../..";
        const handleModule = (_mod, assets)=>{
            if (_mod.layer !== _constants.WEBPACK_LAYERS.server) return;
            const mod = _mod;
            if (!mod.resource) return;
            if (!mod.resource.startsWith(this.appDir + _path.default.sep)) return;
            if (!/\.(js|jsx|ts|tsx|mjs)$/.test(mod.resource)) return;
            const IS_LAYOUT = /[/\\]layout\.[^./\\]+$/.test(mod.resource);
            const IS_PAGE = !IS_LAYOUT && /[/\\]page\.[^.]+$/.test(mod.resource);
            const relativePathToApp = _path.default.relative(this.appDir, mod.resource);
            const relativePathToRoot = _path.default.relative(this.dir, mod.resource);
            const typePath = _path.default.join("types", "app", relativePathToApp.replace(/\.(js|jsx|ts|tsx|mjs)$/, ".ts"));
            const relativeImportPath = _path.default.join(_path.default.relative(typePath, ""), relativePathToRoot.replace(/\.(js|jsx|ts|tsx|mjs)$/, "")).replace(/\\/g, "/");
            const assetPath = assetPrefix + "/" + typePath.replace(/\\/g, "/");
            if (IS_LAYOUT) {
                assets[assetPath] = new _webpack.sources.RawSource(createTypeGuardFile(mod.resource, relativeImportPath, {
                    type: "layout"
                }));
            } else if (IS_PAGE) {
                assets[assetPath] = new _webpack.sources.RawSource(createTypeGuardFile(mod.resource, relativeImportPath, {
                    type: "page"
                }));
            }
        };
        compiler.hooks.compilation.tap(PLUGIN_NAME, (compilation)=>{
            compilation.hooks.processAssets.tap({
                name: PLUGIN_NAME,
                stage: _webpack.webpack.Compilation.PROCESS_ASSETS_STAGE_OPTIMIZE_HASH
            }, (assets)=>{
                for (const entrypoint of compilation.entrypoints.values()){
                    for (const chunk of entrypoint.chunks){
                        compilation.chunkGraph.getChunkModules(chunk).forEach((mod)=>{
                            handleModule(mod, assets);
                        });
                    }
                }
            });
        });
    }
}
exports.FlightTypesPlugin = FlightTypesPlugin;

//# sourceMappingURL=flight-types-plugin.js.map